/** 
* @projectDescription	Globally-used javascript
*
* @id	global.js
*/

/*global document, Prototype, Class, $, $$, Effect, window, Draggable, Event, Position, Element, Form, Ajax, navigator, setTimeout, clearTimeout, swfobject */

/**
 * Adds a delay to mouseenter/mouseleave events for any type of element
 * @alias HoverDelay
 * @param {string} trigger This is the mouseenter/mouseleave trigger
 * @param {array} options This is a set of options that can be passed such as a close selector, enter / leave functions, and delay
 */
var HoverDelay = Class.create({
	initialize : function (trigger, options) {
		this.options = Object.extend({
			closeSelector : '.close', 
			enterCb : function () {},	
			leaveCb : function () {},	
			delay : 0.5
		}, options || {});
		
		this.trigger = $(trigger);
		this.timeout = null; 
		this.active = false;
		this.setup();
	},
	setup : function () {
		var eEvt, lEvt;
		eEvt = this.open.bindAsEventListener(this);
		lEvt = this.close.bindAsEventListener(this);
		this.trigger.observe('mouseenter', eEvt);
		this.trigger.observe('mouseleave', lEvt);
		this.trigger.observe('hoverdelay:stop', function () {
			this.trigger.stopObserving('mouseenter', eEvt);
			this.trigger.stopObserving('mouseleave', lEvt);
		}.bind(this));
		document.observe('pop:active', function () { 
			this.inactive = true; 
		}.bind(this));
		document.observe('pop:inactive', function () { 
			this.inactive = false;
		}.bind(this));
	}, 
	open : function (event) {
		if (this.inactive) { 
			return;
		}
		this.timeout = function () {
			this.trigger.addClassName("active");
			this.options.enterCb.bind(this)();
			this.active = true;
		}.bind(this).delay(this.options.delay);
	},
	close : function (ev) {
		if (this.inactive) { 
			return;
		}
		if (this.timeout) {
			window.clearTimeout(this.timeout);
			this.timeout = null;
		}
		if (this.active) {
			this.options.leaveCb.bind(this)(ev);
			this.active = false;
			this.trigger.removeClassName("active");
		}
	}
});


/**
 * 
 * @alias Kickout
 */
var Kickout = Class.create({
	initialize: function (trigger, container) {
		this.trigger = $(trigger);
		this.container = $(container);
		this.setup();
	},
	setup: function () {
		var hd = new HoverDelay(this.trigger, {
			delay: 0.2,
			enterCb: this.open.bindAsEventListener(this),
			leaveCb: this.close.bindAsEventListener(this)
		});
	},
	open: function (event) {
		this.container.show();
		if (event) {
			event.stop();
		}
	},
	close: function (event) {
		this.container.hide();
		if (event) {
			event.stop();
		}
	}
});


/**
 * 
 * @alias AjaxKickout
 */
var AjaxKickout = Class.create({
	initialize: function (trigger, container) {
		this.trigger   = $(trigger);
		this.container = $(container);
		this.setup(); 
	},
	setup: function () {
		this.trigger.observe('click', this.open.bindAsEventListener(this));
		document.observe('AjaxKickout:toggled', this.test.bindAsEventListener(this));
	},
	open: function (event) {
		if (!this.form) {
			this.form     = this.container.down('form');
			this.closebtn = this.container.down('a.close');
			this.resetbtn = this.container.down('a.reset');
			this.response = this.container.down('div.response');
			this.footer   = this.container.down('div.footer');
			this.closelistener  = this.close.bindAsEventListener(this);
			this.resetlistener  = this.reset.bindAsEventListener(this);
			this.submitlistener = this.submit.bindAsEventListener(this);
		}
		
		this.container.setStyle({ left: '5px', top: '-11px', display: 'block', visibility: 'visible' });
		this.closebtn.observe('click', this.closelistener);
		this.resetbtn.observe('click', this.resetlistener);
		this.form.observe('submit', this.submitlistener);
		document.fire('AjaxKickout:toggled', { id: this.container.id });
		event.stop();
	},
	close: function (event) {
		this.container.hide();
		this.reset(false);
		event.stop();
	},
	test: function (event) {
		if (event.memo.id !== this.container.id) {
			this.close(event);
		}
	},
	reset: function (event) {
		if (this.form) {
			this.response.hide();
			this.footer.hide();
			this.form.show();
		}
		
		if (event) {
			event.stop();
		}
	},
	submit: function (event) {
		this.form.hide();
		this.response.show();
		this.footer.show();
		event.stop();
	}
});


/**
 * 
 * @alias ClientLogin
 */
var ClientLogin = Class.create(AjaxKickout, {
	submit: function ($super, event) {
		this.response.innerHTML = 'Signing in...';
		var ajax = new Ajax.Request('/ajax_client_login/', {
			method:     'post',
			parameters: Form.serialize(this.form),
			onSuccess:  function () { 
				this.response.innerHTML = 'Sorry, we can\'t authenticate you. Please try again.'; 
			}.bind(this) 
		});
		$super(event);
	}
});


/**
 * 
 * @alias sendTrackingEvent
 */
function sendTrackingEvent(event) {
	if (typeof(s) !== "object") { 
		return;
	}
	switch (event) {
		case 'event3':
			s.pageName = 'Newsletter Signup';
			s.events   = 'event3';
			void(s.t());
			break;
	}
}


/**
 * 
 * @alias NewsletterSignup
 */
var NewsletterSignup = Class.create(AjaxKickout, {
	submit: function ($super, event) {
		this.response.innerHTML = 'Sending...';
		var params, ajax;
		params = {
			email:  this.form.elements['email'].value,
			source: document.location.href
		};
		ajax = new Ajax.Request('/ajax_newsletter_signup/', {
			method:     'post',
			parameters: params,
			onSuccess:  function (req) {
				if (req.responseText === 'SUCCESS') {
					this.response.innerHTML = 'Thanks for signing up for the newsletter!';
					sendTrackingEvent('event3');
				} else {
					this.response.innerHTML = 'There was a problem with your email. Please check to make sure you have entered it correctly and try again.';
				}
			}.bind(this)});
		$super(event);
	}
});



/**
 * Pop controller that handles various options such as centering, fade, modal, dragging
 * @alias PopUp
 * @param {string} trigger This is the mouse click trigger
 * @param {string} pop This is the popup
 * @param {array} options This is a set of options that can be passed such as a modal, center, fade, close selector, handle selector (for dragging), and extra open / close methods that can be passed functions
 */
var PopUp = Class.create({
	initialize : function (trigger, pop, options) {
		this.trigger = trigger;
		this.pop = $(pop);
		
		this.options = Object.extend({
			modal:  false,
			centered: false,
			fade: false,
			closeSelector:  '.close',
			handleSelector: '.overlay_head',
			onOpen:    Prototype.emptyFunction,
			onClose:   Prototype.emptyFunction
		}, options || {});
		
		PopUp.i = 0;
		PopUp.open = false;
		this.setup();
	},
	setup : function () {
		this.pop.hide();
		this.trigger.observe('click', this.open.bindAsEventListener(this));
	},
	open : function (ev) {
		
		// allow only one popup opened at a time
		if (PopUp.open) { 
			PopUp.open.close(false); 
		}
		PopUp.open = this;	
	
		document.fire("pop:active");
		this.toTop();
		this.trigger.addClassName("active");
		
		// close button(s)
		this.pop.select(this.options.closeSelector).each(function (el) {
			el.observe('click', this.close.bind(this));
		}.bind(this));
		
		// draggable handle
		this.pop.select(this.options.handleSelector).each(function (el) {
			this.draggable = new Draggable(this.pop, { handle: this.handle, starteffect: false, endeffect: false });
		}.bind(this));
		
		// close pop if user clicks anywhere in document
		this.close_listener = this.close.bindAsEventListener(this);
		Event.observe(document, 'click', this.close_listener);
		document.observe('pop:close',  this.close_listener);
		
		this.pop.observe('mouseenter', function () {
			Event.stopObserving(document, 'click', this.close_listener); 
		}.bind(this));

		this.pop.observe('mouseleave', function () {
			Event.observe(document, 'click', this.close_listener); 
		}.bind(this));
		
		// modal version
		if (this.options.modal) {
			this.initModalWindow('modal_overlay');
		}
		
		// centered version
		if (this.options.centered) {
			this.center();
		}

		// fade effect when appearing
		if (this.options.fade) {
			this.pop.appear({duration: 0.2});
		}
		// else just show
		else {
			this.pop.show();
		}
		
		(this.options.onOpen || Prototype.emptyFunction)();

		if (typeof(ev) === 'object') {
			ev.stop();
		}
	},
	close : function (ev) {
		document.fire("pop:inactive");
		PopUp.open = false;

		this.trigger.removeClassName("active");
		if (this.options.modal) {
			$('modal_overlay').hide();
		}
		if (this.options.fade) {
			this.pop.fade({duration: 0.2});
		}
		else {
			this.pop.hide();
		}
			
		Event.stopObserving(document, 'click', this.close_listener);
		document.stopObserving('pop:close',  this.close_listener);
		this.pop.stopObserving('mouseenter');
		this.pop.stopObserving('mouseleave');
		
		(this.options.onClose || Prototype.emptyFunction)();
		
		if (ev) {
			ev.stop();
		}
	},
	initModalWindow: function (el) {
		$(el).setStyle({
			height: $$('body').first().getHeight() + "px",
			zIndex: 100
		});
		$(el).show();
	},
	toTop: function () {
		PopUp.i += 1;
		this.pop.style.zIndex = PopUp.i + 1000;
		this.pop.show();
	},
	center: function () {

		if (this.hasBeenCentered) { 
			return;
		}
		var w, h, pw, ph, ws;
		w = this.pop.offsetWidth;
		h = this.pop.offsetHeight;
		Position.prepare();
		ws = this.getWindowSize();
		pw = ws[0];
		ph = ws[1];
		this.pop.setStyle({
			top: (ph / 2) - (h / 2) +  Position.deltaY + "px",
			left: (pw / 2) - (w / 2) +  Position.deltaX + "px"
		});
		//this.hasBeenCentered = true;
	},
	getWindowSize: function (w) {
		w = w ? w : window;
		var width, height;
		width = w.innerWidth || (w.document.documentElement.clientWidth || w.document.body.clientWidth);
		height = w.innerHeight || (w.document.documentElement.clientHeight || w.document.body.clientHeight);
		return [width, height];
	},
	toggleSelects: function () {
		if (Prototype.Browser.IE6) {
			$$('select').each(function (e) {
				$(e).toggle();
			});
		}
	}
});

/**
 *
 * @alias Gallery
 */
var Gallery = Class.create({
	initialize: function (container) {
		this.container = $(container);
		
		this.client = this.container.className;
		
		this.clientPath = "/img/work/" + this.client + "/";
		
		this.thumbPath = this.clientPath + "thumb/";
		this.lrgPath = this.clientPath + "lrg/";
		this.overlay = $('overlayEnlarged');
		this.pop = new PopUp($('enlarge'), this.overlay, {centered: true});
		this.thumbs = this.container.down('#thumbnails');
		this.togglers = this.thumbs.select('li.thumb');
		this.prevBtns = [this.container.down('.prev'), this.overlay.down('.prev')];
		this.nextBtns = [this.container.down('.next'), this.overlay.down('.next')];		
		this.currentThumb = this.container.down('.currentPhoto img');
		
		this.loader = this.overlay.down('.loader');
		this.current = 0;
		this.popOpen = false;
		this.loaded = false;
		this.setup();
	},
	setup: function () {
		
		
		this.togglers.each(function (el, i) {
			el.observe('click', this.goTo.bind(this, i));
		}.bind(this));
		
		if (this.prevBtns.length > 1) {
			this.prevBtns.each(function (el) {
				el.observe('click', this.prev.bindAsEventListener(this));
			}.bind(this));
		}
		if (this.nextBtns.length > 1) {
			this.nextBtns.each(function (el) {
				el.observe('click', this.next.bindAsEventListener(this));
			}.bind(this));
		}
		
		document.observe('pop:active', function () { 
			this.popOpen = true;
			this.goTo(this.current);
		}.bind(this));			
		document.observe('pop:inactive', function () { 
			this.popOpen = false;
			this.goTo(this.current);
		}.bind(this));		
	},
	goTo: function (i) {
		this.togglers.invoke('removeClassName', 'active');
		this.togglers[i].addClassName('active');
		
		if (!this.popOpen) {
			this.currentThumb.src = this.thumbPath + "work-thumb-" + (i + 1) + ".jpg";
			this.currentThumb.hide();
			this.currentThumb.appear({duration: 0.3});
		}
		
		this.photoDiv = this.overlay.down('.currentPhoto');
		this.currentPhoto = new Element('img').hide();
		this.photoDiv.update(this.currentPhoto);

		
		if (!this.loaded) {
			this.currentPhoto.observe('load', function () {
				this.currentPhoto.appear({duration: 0.5});
			}.bind(this));
		
			setTimeout(function () {
				this.currentPhoto.writeAttribute('src', this.lrgPath + "work-" + (i + 1) + ".jpg");
				this.loader.hide();
			}.bind(this), 1000);
			this.loaded = true;
		}
		else {
			this.currentPhoto.writeAttribute('src', this.lrgPath + "work-" + (i + 1) + ".jpg");
			this.currentPhoto.appear({duration: 0.3});
			this.loader.hide();
		}
		
		this.current = i;
	},
	prev: function (ev) {
		if (this.current === 0) {
			this.goTo(this.togglers.length - 1);
		}	
		else {
			this.goTo(this.current - 1);
		}
		if (ev) {
			ev.stop();
		}
	},
	next: function (ev) {
		if (this.current === this.togglers.length - 1) {
			this.goTo(0);
		}
		else {
			this.goTo(this.current + 1);
		}
		if (ev) {
			ev.stop();
		}
	}
});

/**
 * Controls the menu kickouts that appear for each client logo on the work landing page
 * @alias AiWork
 */
var AiWork = Class.create({
	initialize: function (container) {
		this.container = $(container);
		this.clients = this.container.select('li.menu');
		this.setup();
	},
	setup: function() {
		var effect;
		this.clients.each(function (el) {
			el.observe('mouseenter', function () {
				el.addClassName('active');
				effect = new Effect.Appear(el.down('em'), { duration: 0.2 });
			});
			el.observe('mouseleave', function () {
				el.removeClassName('active');
				effect = new Effect.Fade(el.down('em'), { duration: 0.1 });
			});
		}.bind(this));
	}
});

/**
 * Highlights clients with the corresponding service offered on the clients landing page
 * @alias AiClients
 */
var AiClients = Class.create({
	initialize: function (service) {
		this.container = $('clients');
		this.clients = this.container.select('.column li');
		this.service = service;
		this.services = this.container.down('#services').select('li a');
		this.bgColor = '#dbdbdb';
		this.setup();
	},
	setup: function() {
		this.services.each(function (el) {
			el.observe('mouseenter', this.highlight.bindAsEventListener(this, el.id));
		}.bind(this));
		this.services.each(function (el) {
			el.observe('mouseleave', this.dehighlight.bindAsEventListener(this));
		}.bind(this));
	},
	highlight: function(ev, service) {
		this.clients.each(function (el) {
			if (el.hasClassName(service)) {
				el.style.backgroundColor = this.bgColor;
			}
		}.bind(this));
		
		if (ev) {
			ev.stop();
		}
	},
	dehighlight: function(ev) {
		this.clients.each(function (el) {
			el.style.backgroundColor = 'transparent';
		}.bind(this));
		
		if (ev) {
			ev.stop();
		}
	}
});

/**
 * 
 * @alias AiCompany
 */
var AiCompany = Class.create({
	initialize: function (container) {
		this.container = $(container);
		this.peeps = this.container.select('li');
		
		this.setup();
	},
	setup: function () {
		this.peeps.each(function (el) {
			el.observe('click', function () {
				var link = el.select('a').first();
				window.location = link.href;
			}.bind(this));
			el.observe('mouseenter', function () { 
				this.addClassName('active'); 
			});
			el.observe('mouseleave', function () { 
				this.removeClassName('active'); 
			});
		});
	}
});


 

/*--------------------------------------------------------------------------*/

document.observe('dom:loaded', function () {
	// cache background images in ie6 to prevent multiple http requests
	if (Prototype.Browser.IE6) {
		try {
			document.execCommand('BackgroundImageCache', false, true);
		} catch (e) {}
	}

	var clientLogin,
		newsletterSignup,
		gallery,
		projectKickout,
		workMenus,
		clients,
		company;
	
	if ($('client-login')) {
		clientLogin = new ClientLogin('client-login-trigger', 'client-login');
	}
	
	if ($('newsletter-signup')) {
		newsletterSignup = new NewsletterSignup('newsletter-signup-trigger', 'newsletter-signup');
	}
	
	if ($('news')) {
		if ($('leftColumn')){
			if ($('leftColumn').getHeight() > $('rightColumn').getHeight()) {
				$('rightColumn').style.height = $('leftColumn').getHeight() + "px";
			}
			else {
				$('leftColumn').style.height = $('rightColumn').getHeight() + "px";
			}
		}
	}
	
	if ($('rightColumn')) {
		if ($('rightColumn').hasClassName('slideshow')) {
			gallery = new Gallery('sidebar');
		}
	}
	
	if ($('rightColumn')) {
		if ($('rightColumn').hasClassName('listView')) {
			company = new AiCompany('rightColumn');
		}
	}
	
	if($('clients')) {
		workMenus = new AiWork('clients');
	}	
	
	if($('selectProject')) {
		projectKickout = new Kickout('selectProject', 'projects');
	}	
	
	if($('clients') && $('services')) {
		clients = new AiClients('clients');
	}
});
